#! /usr/bin/env julia
# ---------------------------------------------------------


# ---------------------------------------------------------
using CSV, FileIO
using DataFrames
using Parameters, Distributed
using Statistics: mean, std, cor, I, kron, UpperTriangular, median
using Plots, LaTeXStrings
using JLD2
pgfplotsx(size = (1400, 900))

push!(LOAD_PATH, "./src")
using project_routines
# ---------------------------------------------------------


# ---------------------------------------------------------
# logging
run(`mkdir -p log`)
using Logging, LoggingExtras, Dates

# clean up the files
map(x->rm(x, force=true), ["./log/table10_info.log.jl", "./log/table10_warn.log.jl", "./log/table10_full.log.jl"]);
# custom timestamps
const log_date_format = "yyyy-mm-dd HH:MM:SS"
timestamp_logger(logger) = TransformerLogger(logger) do log
  merge(log, (; message = "$(Dates.format(now(), log_date_format)) \n$(log.message)"))
end

# create the debugger
demux_logger = TeeLogger(
    MinLevelLogger(FileLogger("./log/table10_info.log.jl"), Logging.Info),
    MinLevelLogger(FileLogger("./log/table10_warn.log.jl"), Logging.Warn),
    MinLevelLogger(FileLogger("./log/table10_full.log.jl"), Logging.Debug),
    ConsoleLogger(stdout, Logging.Info),   # Common logger to be able to see info and warning in repl
) |> timestamp_logger |> global_logger;
# ---------------------------------------------------------


# ---------------------------------------------------------
## Load in the data
calibration_parameters = CSV.read("./input/param_calibration.csv", DataFrame);
Lj = CSV.read("./input/Ljstar.csv", DataFrame).Lj;
PjCj = CSV.read("./input/PjCj.csv", DataFrame);
Tau = CSV.read("./input/Taustar.csv", DataFrame) |> Matrix;
mij_data = CSV.read("./input/mij_data.csv", DataFrame) |> Matrix;
mii_data = CSV.read("./input/mii_data.csv", DataFrame) |> Matrix;
mij_share = CSV.read("./input/mij_share.csv", DataFrame) |> Matrix;
mii_share = CSV.read("./input/mii_share.csv", DataFrame) |> Matrix;
gdpdata_vec = CSV.read("./input/gdpdata_df.csv", DataFrame) |> Matrix;
# ---------------------------------------------------------


# ---------------------------------------------------------
# Get calibration parameters
eta, psi, sigma, gamma, phi, years, dt, BURN_IN =  Vector(calibration_parameters[1, :]);
N = size(Lj, 1);
t = (0:dt:years-dt);
T = Int(size(t, 1))

Pj_lambda = PjCj.Pj_lambda; 
Cj_lambda = PjCj.Cj_lambda;
lambdaj_pp_gamma0 = Pj_lambda ./ Cj_lambda .^ (-gamma);
lambdaj_pp_gamma = lambdaj_pp_gamma0 ./ sum(lambdaj_pp_gamma0);

## Generate the random state path
states = random_productivity_state(N, t, sigma, psi, years, dt, BURN_IN);
# ---------------------------------------------------------


# ---------------------------------------------------------
@info "Complete Markets Simulation"
calibrationComplete = parameters(eta, gamma, N, dt, Lj, lambdaj_pp_gamma, Tau, 0.2, 1e-5);
uvars_complete = utilityvars_complete(N=N, T=T, dt=dt);

Complete_output = solver_CompleteMarkets(calibrationComplete, states, uvars_complete)
# ---------------------------------------------------------


# ---------------------------------------------------------
@info "Autarky Simulation"
# Solve an Autarky version of the model to extract moments and Pareto weights
calibrationAut = parameters(eta, gamma, N, dt, Lj, lambdaj_pp_gamma, Tau, 0.2, 1e-5);
uvars_autarky = utilityvars_autarky(N=N, T=T, dt=dt);

lambdaj_ce_scld, Autarky_output = solver_CE(calibrationAut, states, uvars_autarky; 
  Analysis_out = true)
# ---------------------------------------------------------


# ---------------------------------------------------------
@info "Incomplete Markets Simulation"
## Pareto weights of the incomplete markets model
# The Pareto weights of incomplete market are a linear combination of
#   Autarky and Complete market's constant weights
lambdaj = lambdaj_ce_scld * (1.0 - phi) .+ lambdaj_pp_gamma * phi;
calibrationIncomplete = parameters(eta, gamma, N, dt, Lj, lambdaj, Tau, 0.2, 1e-5);
uvars_incomplete = utilityvars_incomplete(N=N, T=T, dt=dt);

~, dict_out = solver_incomplete(calibrationIncomplete, states, uvars_incomplete);
Benchmark_output = dict_out["Incomp_output"]
# ---------------------------------------------------------


# ---------------------------------------------------------
@info "Saving results"
## Table 10 of the paper
Sim_out = DataFrame(
        Variable =  ["Consumption volatility", "Currency volatility", "Cross-country consumption correlation",
          "Cross-country output correlation", "Backus-Smith coefficient"],
        Autarky = vec(Autarky_output),
        Incomplete_Markets = vec(dict_out["Incomp_output"]),
        Complete_Markets = vec(Complete_output),
    );
@info "Table 10 in the paper"
@info Sim_out

# Save the table
CSV.write("./output/Table_10.csv", Sim_out)
save("./tmp/simulation_dict.jld2", dict_out)
# ---------------------------------------------------------


# ---------------------------------------------------------
# Actually generate table 10 full tabular environment
table_header = LaTeXString(raw"""
\begin{tabular}{lcccc}
\toprule
""")

table_body_calibration = L"""

\multicolumn{5}{c}{Panel A: Calibration} \\
\midrule
Description & Parameter & Value & \multicolumn{2}{c}{Target/Source} \\
\midrule
Elasticity between goods & $\eta$ & %$eta    & \multicolumn{2}{c}{\citet{AndersonvanWincoop04}} \\
Mean reversion of productivity & $\kappa$ & %$psi & \multicolumn{2}{c}{Output gap half-life} \\
Risk aversion & $\gamma$ & %$gamma     & \multicolumn{2}{c}{\multirow{2}[1]{*}{Consumption and currency volatility}} \\
Productivity volatility & $\sigma$ & %$sigma  & \multicolumn{2}{c}{} \\
\midrule
"""

table_body_moments = L"""
\multicolumn{5}{c}{Panel B: Moments (Annual)} \\
\midrule
      &       & Financial &       & Complete \\
      & Data  & Autarky &       & Markets \\
      & (90\% CI) & $\phi=0$ & $\phi=0.61$ & $\phi = 1$ \\
\cmidrule{2-5}Consumption volatility     & (0.007, 0.040)  & 
  %$(Sim_out[1,2]) & %$(Sim_out[1,3]) & %$(Sim_out[1,4]) \\
Currency volatility                  & (0.067, 0.291)      &    
  %$(Sim_out[2,2]) & %$(Sim_out[2,3]) & %$(Sim_out[2,4]) \\
Cross-country consumption correlation & (-0.425, 0.618)    & 
  %$(Sim_out[3,2]) & %$(Sim_out[3,3]) & %$(Sim_out[3,4]) \\
Cross-country output correlation      & (-0.162, 0.758)    & 
  %$(Sim_out[4,2]) & %$(Sim_out[4,3]) & %$(Sim_out[4,4]) \\
Backus-Smith coefficient             & (-3.715, 1.867)     & 
  %$(Sim_out[5,2]) & %$(Sim_out[5,3]) & %$(Sim_out[5,4]) \\

\bottomrule
\end{tabular}
"""

# Export result to .tex file
write("./output/table_10.tex", 
  table_header * table_body_calibration * table_body_moments);
# ---------------------------------------------------------


# ---------------------------------------------------------
## Figure 4 and Figure 5. Results are saved as PDF automatically
# function Fig4Fig5(dict_out)
@info "Generating Figures 4 and 5"

default(legend = :false)
var_base_org = std(dict_out["basej"], dims = 2);
rho_ij_org = dict_out["Sigma"] ./ (repeat(var_base_org, 1, N) + repeat(var_base_org', N, 1));
dpii_sd = std(diff(log.(dict_out["pj"]), dims = 2), dims = 2);
phi_ij_org =
        dict_out["mean_mij"][:, :, 1] ./ mean(dict_out["pj"], dims = 2)' .*
        dpii_sd';
phi_ij_tr = dict_out["mij"] .* reshape(dpii_sd ./ dict_out["pj"], 1, N, :);
phibar = mean(phi_ij_tr, dims = 1);
# ---------------------------------------------------------


# ---------------------------------------------------------
# Fig 4: calibration performance
@info "Generating Figures 4"
X, Y = 100 * mij_data[Int.(I(N)).==0], 100 * mij_share[Int.(I(N)).==0]
p1_a = myplotter_draft(;
  X = X, Y = Y,
  rho_loc = [20, 30],
  xx_line_bound = [0, 40],
  y_label = "Model",
  title = "Import Share (pct.)" );

X, Y = 100 * mii_data, 100 * mii_share
p1_b = myplotter_draft(;
  X = X, Y = Y,
  rho_loc = [20, 60],
  xx_line_bound = [0, 90],
  title = "Domestic Absorption (pct.)" );

X, Y = Lj, gdpdata_vec
p1_c = myplotter_draft(;
  X = X, Y = Y,
  rho_loc = [0.45, 0.7],
  xx_line_bound = [0, 1],
  y_label = "Model",
  x_label = "Data",
  title = "GDP (US = 1)" );

X = log.((reshape(Lj, N, 1).*mij_data)[Int.(I(N)).==0]);
Y = log.((gdpdata_vec.*mij_share)[Int.(I(N)).==0]);
p1_d = myplotter_draft(;
  X = X, Y = Y,
  rho_loc = [-10, -6],
  xx_line_bound = [-12.5, -2.5],
  x_label = "Data",
  title = "Log Imports, in USD" );

fig4_dr = plot(p1_a, p1_b, p1_c, p1_d, layout = (2, 2), size = (900, 900));
# fig4_dr
# ---------------------------------------------------------


# ---------------------------------------------------------
# Fig 5: Main Model Predictions
@debug "Generating Figures 5"
X, Y = Tau[Int.(I(N)).==0], dict_out["beta"][Int.(I(N)).==0];
p2_a = myplotter_draft(;
  X = X, Y = Y,
  reg_line = true,
  x_label = L"\tau_{ij}",
  y_label = L"\beta_{ij}",
  title = "Main Calibration Results");
ylims!(-2, 3);

# Tau and phi (our trade exposure measure)
X = Tau[Int.(I(N)).==0]
Y = log.(phi_ij_org[Int.(I(N)).==0])
p2_b = myplotter_draft(;
  X = X, Y = Y,
  reg_line = true,
  x_label = L"\tau_{ij}",
  y_label = L"{log}~~\varphi_{ij}");

# TAU / R2
X, Y = no_diag_mean(Tau, 2), no_diag_mean(dict_out["R2"], 2)[:];
p2_c = myplotter_draft(;
  X = X, Y = Y,
  reg_line = true,
  x_label = L"$\overline{\tau}_{j}$",
  y_label = L"$\overline{R^2_{j}}$" );

# Phi / R2, # ϕ outward and R2 outward
X = (N - 1) * no_diag_mean(phi_ij_org, 2)[:];
Y = no_diag_mean(dict_out["R2"], 2)[:];
p2_d = myplotter_draft(;
  X = X, Y = Y,
  reg_line = true,
  x_label = L"$\sum_{i \neq j}\varphi_{ij}$",
  y_label = L"$\overline{R^2_{j}}$"  );

fig5_dr = plot(p2_a, p2_b, p2_c, p2_d,
  layout = (2, 2),
  size = (900, 900),
  title = ["Panel A" "Panel B" "Panel C" "Panel D"]    );
# fig5_dr
# ---------------------------------------------------------
   

# ---------------------------------------------------------
# SAVE THE FIGURES
@info "Saving Figures"

try
  savefig(fig4_dr, "./output/fig4.pdf")
  savefig(fig4_dr, "./output/fig4.tex")
  savefig(fig5_dr, "./output/fig5.pdf")
  savefig(fig5_dr, "./output/fig5.tex")
catch
  # In case the required PDF engine is not installed
  @warn "\nCannot convert to PDF \n... trying png ...\n"
  try
    png(fig4_dr, "./output/fig4.png")
    png(fig5_dr, "./output/fig5.png")
  catch
    @warn "Cannot print the figures"
  end
end

# for publication eps or ps (svg is probably superior)
savefig(fig4_dr, "./output/fig4.svg")
savefig(fig5_dr, "./output/fig5.svg")'
# - if inkscape is installed
run(`inkscape ./output/fig4.svg -o ./output/fig4.eps --export-ignore-filters --export-ps-level=3`)
run(`inkscape ./output/fig5.svg -o ./output/fig5.eps --export-ignore-filters --export-ps-level=3`)
# ---------------------------------------------------------

